"use strict";

(function () {
    let app = angular.module('workstation', ['workstationResearcher', 'workstationHrManager', 'workstationSchedule']);

    app.directive('workstationEmpty', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/workstation-empty.html',
            controller: function ($rootScope, $scope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;

                ctrl.employeeSets = $rootScope.settings.unassignedEmployees.map(employee => {
                    return {
                        floor: null,
                        employee: employee,
                        status: 'Unassigned'
                    }
                });

                ctrl.placeEmployee = employee => {
                    _.remove($rootScope.settings.unassignedEmployees, employee);
                    $rootScope.selectedWorkstation.employee = employee;
                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                }
            },
            controllerAs: "ctrl"
        };
    });

    app.directive('workstationEmployee', function () {
        return {
            restrict: 'E',
            scope: {
                'closeAllUi': '=',
            },
            templateUrl: 'templates/workstation-employee.html',
            link: function () {
                dragscroll.reset();
            },
            controller: function ($rootScope, $scope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                $scope.EmployeeTypeNames = EmployeeTypeNames;
                ctrl.employeeProductivity = CalculateWorkstationProductivity($rootScope.selectedWorkstation);
                ctrl.hasEmployeeHrManager = HasEmployeeHrManager($rootScope.selectedWorkstation.employee);
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'selectedWorkstation');
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'companyTier');
                let selectedWorkstation = $rootScope.selectedWorkstation;
                Helpers.MovePropToIsolatedScope($rootScope, $scope, 'daysPlayed');

                $scope.$on('$destroy', (e) => {
                    if (selectedWorkstation.employee) {
                        selectedWorkstation.employee.lastTab = ctrl.tab;
                        if (selectedWorkstation.employee.name.length == 0) {
                            selectedWorkstation.employee.name = selectedWorkstation.employee.originalName;
                        }
                    }
                });

                Helpers.RootScopeWatchAndDestroy($scope, 'selectedWorkstation', (newValue) => {
                    if (newValue == null || newValue.employee == null) return;

                    $scope.EmployeeTypes = EmployeeTypeNames;

                    ctrl.isProductionEmployee = [EmployeeTypeNames.Developer, EmployeeTypeNames.DevOps, EmployeeTypeNames.Designer, EmployeeTypeNames.Marketer]
                        .includes($rootScope.selectedWorkstation.employee.employeeTypeName);

                    this.getInitialTab = function () {
                        switch ($rootScope.selectedWorkstation.employee.employeeTypeName) {
                            case EmployeeTypeNames.Researcher:
                                return "features";
                            case EmployeeTypeNames.Manager:
                                return "manager";
                            case EmployeeTypeNames.HrManager:
                                return "hrmanager";
                            case EmployeeTypeNames.SalesExecutive:
                                return "sales-executive";
                            case EmployeeTypeNames.LeadDeveloper:
                                return "leadDeveloper";
                            default:
                                if (ctrl.isProductionEmployee) return "development";
                                return "stats";
                        }
                    };

                    ctrl.tab = selectedWorkstation.employee.lastTab || this.getInitialTab();
                    $scope.isManager = $rootScope.selectedWorkstation.employee != null && $rootScope.selectedWorkstation.employee.employeeTypeName == EmployeeTypeNames.Manager;
                    $scope.isHrManager = $rootScope.selectedWorkstation.employee != null && $rootScope.selectedWorkstation.employee.employeeTypeName == EmployeeTypeNames.HrManager;

                    ctrl.sendOnVacation = () => {
                        let message = `<p>${ Helpers.GetLocalized('confirm_vacation', {
                            name: $rootScope.selectedWorkstation.employee.name,
                            days: Configuration.VACATION_DAYS
                        }) }</p>`;
                        let contract = null;

                        if ($rootScope.selectedWorkstation.employee.employeeTypeName == EmployeeTypeNames.SalesExecutive && $rootScope.selectedWorkstation.employee.contractId != null) {
                            contract = $rootScope.settings.contracts.find(x => x.id == $rootScope.selectedWorkstation.employee.contractId);
                            message += `<p class="color-red text-bold">${ Helpers.GetLocalized('contract_cancelation_warning') }</p>`;
                        }

                        $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), message, () => {
                            $rootScope.selectedWorkstation.employee.vacationHoursLeft = Configuration.VACATION_HOURS;
                            $rootScope.selectedWorkstation.employee.lastVacationDay = $rootScope.daysPlayed + (Configuration.VACATION_HOURS / 24);
                            wallhack.sendEvent("vacation", `Mood: ${ Math.round($rootScope.selectedWorkstation.employee.mood) }%`);

                            if (contract != null) { // Sales Executive only
                                Helpers.CancelContract(contract);
                            }
                            $rootScope.$broadcast(GameEvents.EmployeeChange);
                        });
                    };

                    ctrl.detachFromWorkstation = () => {
                        Helpers.RemoveEmployeeFromWorkstation($rootScope.selectedWorkstation);
                    };

                    ctrl.dismiss = () => {
                        Helpers.FireEmployee($rootScope.selectedWorkstation.employee);
                    };

                    $scope.$broadcast(GameEvents.EmployeeChange);
                });

                this.controlledByManager = function (employee) {
                    if (employee == null) return 0;

                    return $rootScope.settings.workstations.some(x => x.employee != null && x.employee.employeeTypeName == EmployeeTypeNames.Manager &&
                    x.employee.employees.some(employeeId => employeeId == employee.id));
                }
            },
            controllerAs: "ctrl"
        };
    });

    app.directive('workstationStats', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/workstation/stats.html',
            controller: function ($rootScope, $scope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.CalculateEmploymentDuration = CalculateEmploymentDuration;
                ctrl.CalculateWorkstationProductivity = CalculateWorkstationProductivity;
                ctrl.HasEmployeeHrManager = HasEmployeeHrManager;
                ctrl.officeBonus = $rootScope.officeBonus;
                ctrl.wer = Helpers.CalculateWer();

                let loadView = () => {
                    if(!$rootScope.selectedWorkstation) return;
                    ctrl.employementDuration = CalculateEmploymentDuration($rootScope.selectedWorkstation.employee, $rootScope.settings.date);
                    ctrl.moodBonus = GetMoodPenalty($rootScope.selectedWorkstation);
                    ctrl.managerBonus = GetManagerBonus($rootScope.selectedWorkstation);
                    ctrl.productivity = CalculateWorkstationProductivity($rootScope.selectedWorkstation);
                };

                $scope.$on(GameEvents.OnNewHour, () => loadView());
                Helpers.RootScopeWatchAndDestroy($scope, 'selectedWorkstation.employee.mood', () => loadView());

                this.isReadyForTierUpgrade = () => {
                    let employee = $rootScope.selectedWorkstation.employee;

                    let tierForLevels = Helpers.GetTierForEmployeeLevels();
                    let employeeLevelArray = _.toArray(EmployeeLevels);
                    let nextLevel = employeeLevelArray[employeeLevelArray.indexOf(employee.level) + 1];

                    return tierForLevels[nextLevel] <= Helpers.CalculateCompanyTier();
                };

                this.getUpgradeTooltipText = () => {
                    let employee = $rootScope.selectedWorkstation.employee;
                    let tierForLevels = Helpers.GetTierForEmployeeLevels();
                    let employeeLevelArray = _.toArray(EmployeeLevels);
                    let nextLevel = employeeLevelArray[employeeLevelArray.indexOf(employee.level) + 1];

                    if (tierForLevels[nextLevel] > Helpers.CalculateCompanyTier()) {
                        return Helpers.GetLocalized('tip_upgrade_employee', {tier: tierForLevels[nextLevel]});
                    }
                };

                ctrl.upgradeEmployee = () => {
                    let employee = $rootScope.selectedWorkstation.employee;
                    let employeeLevelArray = _.toArray(EmployeeLevels);
                    let nextLevel = employeeLevelArray[employeeLevelArray.indexOf(employee.level) + 1];

                    $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), Helpers.GetLocalized("confirm_upgrade_employee", {
                        name: employee.name,
                        level: nextLevel
                    }), () => {
                        employee.level = nextLevel;
                        employee.salary += 1500;
                        employee.speed += 15;
                        PlaySound(Sounds.selection);
                    });
                };

                ctrl.isProducer = () => {
                    let employeeTypes = [EmployeeTypeNames.Developer, EmployeeTypeNames.Manager];
                    return employeeTypes.some(x => x == $rootScope.selectedWorkstation.employee.employeeTypeName);
                };

                let loadStats = () => {
                    ctrl.isManager = $rootScope.selectedWorkstation.employee.employeeTypeName === EmployeeTypeNames.Manager;
                    ctrl.managerWorkstation = GetManagerWorkstationByEmployeeId($rootScope.selectedWorkstation.employee.id);
                };

                $scope.$watch('selectedWorkstation', () => loadStats(), true);
            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('workstationLeadDeveloper', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/workstation/leadDeveloper.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.controlledByManager = GetManagerWorkstationByEmployeeId($rootScope.selectedWorkstation.employee.id) != null;

                let loadModules = () => {
                    if ($rootScope.selectedWorkstation == null || $rootScope.selectedWorkstation.employee == null) return;
                    let employeeLevelsArray = _.toArray(EmployeeLevels);

                    ctrl.modules = Components.filter(x =>
                        [ComponentTypes.Module, ComponentTypes.Server].includes(x.type))
                        .map(x => {
                            x.order = employeeLevelsArray.indexOf(x.employeeLevel);
                            x.isResearched = $rootScope.selectedWorkstation.employee.components.some(component => component.name == x.name);
                            return x;
                        });
                };

                let loadTaskInfo = () => {
                    if ($rootScope.selectedWorkstation == null || $rootScope.selectedWorkstation.employee != null) return;
                    ctrl.taskProgress = $rootScope.getTaskPercentage();
                    ctrl.task = $rootScope.selectedWorkstation.employee.task;
                    ctrl.activeTaskStacks = ctrl.task == null ? null : Helpers.ConvertRequirementsIntoStacks(ctrl.task.module.requirements);

                    let employeeStat = Helpers.GetCurrentEmployeeStat();
                    ctrl.isAwaiting = employeeStat.employeeState == 'Awaiting' && employeeStat.employee.task.state == 'Stalled';
                };

                Helpers.RootScopeWatchAndDestroy($scope, 'selectedWorkstation.employee.level', () => loadModules());
                Helpers.RootScopeWatchCollectionAndDestroy($scope, 'settings.inventory', () => loadModules());
                Helpers.RootScopeWatchAndDestroy($scope, 'selectedWorkstation.employee.task', task => loadTaskInfo(), true);
                Helpers.RootScopeWatchAndDestroy($scope, 'selectedWorkstation', () => loadTaskInfo());

                ctrl.clickModule = module => {
                    let researchModule = () => {
                        let totalPrice = Helpers.GetResearchPrice(module);
                        $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), Helpers.GetLocalized('confirm_research_module', {
                            name: Helpers.GetLocalized(module.name),
                            price: numeral(totalPrice).format(Configuration.CURRENCY_FORMAT)
                        }), () => {
                            $rootScope.safeBuy(() => {
                                $rootScope.selectedWorkstation.employee.components.push(Helpers.Clone(module));
                                PlaySound(Sounds.selection);
                                loadModules();
                                $rootScope.$broadcast(GameEvents.SelectedWorkstationChange);
                            }, totalPrice, Helpers.GetLocalized('transaction_module_researched', {
                                name: Helpers.GetLocalized(module.name),
                                employeeName: $rootScope.selectedWorkstation.employee.name
                            }))
                        });
                    };

                    let mergeModule = () => {
                        module.id = chance.guid();

                        let produceHours = Helpers.CalculateComponentProductionHours(module);

                        let hasAllRequirements = Object.keys(module.requirements).every(componentName => {
                            return ($rootScope.settings.inventory[componentName] || 0) >= module.requirements[componentName];
                        });

                        if (hasAllRequirements) {
                            // Spent components
                            Object.keys(module.requirements).forEach(componentName => {
                                $rootScope.settings.inventory[componentName] -= module.requirements[componentName];
                            });

                            $rootScope.selectedWorkstation.employee.task = {
                                module: module,
                                totalMinutes: Math.max(1, (Math.round(produceHours / 3))) * 60,
                                completedMinutes: 0,
                                state: Enums.TaskStates.Running,
                                autoRepeat: $rootScope.selectedWorkstation.employee.task != null
                                    ? $rootScope.selectedWorkstation.employee.task.autoRepeat
                                    : false
                            };

                            $rootScope.$broadcast(GameEvents.EmployeeChange);
                        } else {
                            $rootScope.showMessage('', Helpers.GetLocalized('warning_leaddev_missing_components'))
                        }
                    };

                    if (module.isResearched) { // Merge
                        if ($rootScope.selectedWorkstation.employee.task == null || $rootScope.selectedWorkstation.employee.task.completedMinutes == 0) {
                            if ($rootScope.selectedWorkstation.employee.task != null) {
                                returnComponentsFromTask();
                            }
                            mergeModule();
                        } else {
                            if ($rootScope.selectedWorkstation.employee.task.module.name != module.name) {
                                $rootScope.confirm(``, Helpers.GetLocalized('confirm_module_switch'), () => {
                                    returnComponentsFromTask();
                                    mergeModule();
                                });
                            }
                        }
                    } else { // Research
                        researchModule();
                    }
                };

                ctrl.clickAutoRepeat = () => {
                    if (ctrl.controlledByManager) {
                        $rootScope.selectedWorkstation.employee.task.autoRepeat = !$rootScope.selectedWorkstation.employee.task.autoRepeat
                    } else {
                        $rootScope.showMessage('', Helpers.GetLocalized('auto_repeat_warning'));
                    }
                };

                let returnComponentsFromTask = () => {
                    Object.keys($rootScope.selectedWorkstation.employee.task.module.requirements).forEach(componentName => {
                        $rootScope.settings.inventory[componentName]++;
                    });
                };

                ctrl.cancel = () => {
                    returnComponentsFromTask();
                    $rootScope.selectedWorkstation.employee.task = null;
                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                }
            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('workstationDevelopment', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/workstation/development.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.showOvertimeButton = false;
                ctrl.controlledByManager = GetManagerWorkstationByEmployeeId($rootScope.selectedWorkstation.employee.id) != null;

                ctrl.isComponentAvailable = (employee, component) => {
                    return employee.components.some(x => x.name == component.name);
                };

                let loadComponents = () => {
                    let taskPercentage = $rootScope.getTaskPercentage();
                    let baseComponents = Components.filter(x =>
                    x.type == ComponentTypes.Component &&
                    x.employeeTypeName === $rootScope.selectedWorkstation.employee.employeeTypeName);

                    let availableComponentNames = Helpers
                        .GetComponentsByEmployeeLevel($rootScope.selectedWorkstation.employee.level)
                        .map(x => x.name);

                    let employeeLevelsArray = _.toArray(EmployeeLevels);

                    ctrl.componentsInfo = _.orderBy(baseComponents.map(x => {
                        let selected = $rootScope.selectedWorkstation.employee.task != null &&
                            $rootScope.selectedWorkstation.employee.task.state == Enums.TaskStates.Running &&
                            $rootScope.selectedWorkstation.employee.task.component.name == x.name;

                        return {
                            component: x,
                            selected: selected,
                            order: employeeLevelsArray.indexOf(x.employeeLevel),
                            progress: selected ? taskPercentage : null,
                            isResearched: $rootScope.selectedWorkstation.employee.components.some(c => c.name == x.name),
                            isAvailable: availableComponentNames.includes(x.name)
                        }
                    }), 'order');
                };

                $scope.$watch('selectedWorkstation.employee.overtimeMinutes', () => loadComponents());
                $scope.$watch('selectedWorkstation.employee', () => loadComponents(), true);

                ctrl.workOvertime = () => {
                    $rootScope.selectedWorkstation.employee.overtimeMinutes = ctrl.overtimeMinutes;
                    $rootScope.selectedWorkstation.employee.mood -= 5;
                    ctrl.showOvertimeButton = false;
                };

                ctrl.developComponent = componentInfo => {
                    if (!componentInfo.isAvailable) return;
                    if (componentInfo.isResearched) {
                        let createTask = () => {
                            $rootScope.selectedWorkstation.employee.task = Helpers.GetDevelopmentTask(componentInfo.component);
                            if (ctrl.controlledByManager) {
                                $rootScope.selectedWorkstation.employee.task.autoRepeat = true;
                            }
                            $rootScope.selectedWorkstation.employee.lastProducedComponentName = componentInfo.component.name;
                        };

                        if ($rootScope.selectedWorkstation.employee.task != null && $rootScope.selectedWorkstation.employee.task.completedMinutes > 0) {
                            let message = Helpers.GetLocalized('confirm_module_switch');
                            $rootScope.confirm('', message, () => createTask());
                        } else {
                            createTask();
                        }
                    } else { // Research
                        let totalPrice = Helpers.GetResearchPrice(componentInfo.component);
                        $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), Helpers.GetLocalized('confirm_research_module', {
                            name: Helpers.GetLocalized(componentInfo.component.name),
                            price: numeral(totalPrice).format(Configuration.CURRENCY_FORMAT)
                        }), () => {
                            $rootScope.safeBuy(() => {
                                $rootScope.selectedWorkstation.employee.components.push(Helpers.Clone(componentInfo.component));
                                $scope.$broadcast(GameEvents.EmployeeChange);
                                $rootScope.$broadcast(GameEvents.SelectedWorkstationChange);
                            }, totalPrice, Helpers.GetLocalized('transaction_module_researched', {
                                name: Helpers.GetLocalized(componentInfo.component.name),
                                employeeName: $rootScope.selectedWorkstation.employee.name
                            }))
                        });
                    }

                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                };

                ctrl.cancelDevelopment = () => {
                    $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), Helpers.GetLocalized('confirm_cancel_component'), () => {
                        $rootScope.selectedWorkstation.employee.task = null;
                        $rootScope.selectedWorkstation.employee.overtimeMinutes = 0;
                        $rootScope.$broadcast(GameEvents.EmployeeChange);
                    });
                };
                ctrl.clickAutoRepeat = () => {
                    if (ctrl.controlledByManager) {
                        $rootScope.selectedWorkstation.employee.task.autoRepeat = !$rootScope.selectedWorkstation.employee.task.autoRepeat
                    } else {
                        $rootScope.showMessage('', Helpers.GetLocalized('auto_repeat_warning'));
                    }
                };

            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('workstationManager', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/workstation/manager.html',
            controller: function ($rootScope, $scope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.employeeProductivity = $scope.$parent.ctrl.employeeProductivity;

                let loadView = () => {
                    ctrl.maxInCharge = Helpers.CalculateMaxInCharge($rootScope.selectedWorkstation.employee);
                    let workstations = $rootScope.settings.office.workstations;
                    let managers = workstations.filter(workstation => workstation.employee != null && workstation.employee.employeeTypeName == EmployeeTypeNames.Manager && workstation != $rootScope.selectedWorkstation).map(x => x.employee);

                    let notAvailableEmployeeIds = _.flatten(managers.map(x => x.employees));

                    let allowedEmployeeTypes = [
                        EmployeeTypeNames.Developer,
                        EmployeeTypeNames.Designer,
                        EmployeeTypeNames.SalesExecutive,
                        EmployeeTypeNames.LeadDeveloper,
                        EmployeeTypeNames.Marketer,
                        EmployeeTypeNames.Researcher,
                        EmployeeTypeNames.DevOps
                    ];

                    ctrl.employees = workstations.filter(x =>
                        x.employee != null && allowedEmployeeTypes.some(allowedEmployeeType => x.employee.employeeTypeName == allowedEmployeeType) && !notAvailableEmployeeIds.some(employeeId => x.employee.id == employeeId)
                    );

                    ctrl.bonusPerEmployee = Math.ceil(ctrl.employeeProductivity / $rootScope.selectedWorkstation.employee.employees.length);
                };

                $scope.$watch('selectedWorkstation', () => loadView(), true);

                this.isEmployeeActive = function (manager, employee) {
                    if (manager == null || employee == null || manager.employees == null) return null;

                    return manager.employees.some(id => id == employee.id);
                };

                this.setEmployeeToManager = function (manager, employee) {
                    let exists = this.isEmployeeActive(manager, employee);
                    if (exists) { // Disable
                        if (employee.task != null) employee.task.autoRepeat = false;

                        _.remove(manager.employees, id => {
                            return id == employee.id
                        });
                    } else { // Enable
                        if (manager.employees.length < ctrl.maxInCharge) {
                            manager.employees.push(employee.id);

                            // Enable auto repeat
                            if (employee.task != null) employee.task.autoRepeat = true;
                        }
                    }
                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                }
            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('workstationSalesExecutive', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/workstation/sales-executive.html',
            controller: function ($rootScope, $scope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.controlledByManager = GetManagerWorkstationByEmployeeId($rootScope.selectedWorkstation.employee.id) != null;
                ctrl.taskProgress = 0;
                $scope.Enums = Enums;
                $scope.EmployeeLevels = EmployeeLevels;

                $rootScope.selectedWorkstation.employee.task = $rootScope.selectedWorkstation.employee.task || {
                        contract: null,
                        contractId: null,
                        totalMinutes: 0,
                        completedMinutes: 0,
                        autoRepeat: false,
                        contractType: null,
                        size: null,
                        state: Enums.TaskStates.Stopped
                    };

                $scope.$watch('selectedWorkstation.employee.task.completedMinutes', completedMinutes => {
                    ctrl.taskProgress = (completedMinutes == null || completedMinutes == 0)
                        ? 0
                        : $rootScope.getTaskPercentage();
                });

                $scope.$watch('selectedWorkstation.employee.task.contractId', contractId => {
                    ctrl.activeContract = contractId == null
                        ? null
                        : $rootScope.settings.contracts.find(x => x.id == contractId);
                });

                $scope.$watch('[selectedWorkstation.employee.task.size, selectedWorkstation.employee.task.contractType]', value => {
                    let hours = 0;
                    switch ($rootScope.selectedWorkstation.employee.task.size) {
                        default:
                            hours += 2;
                            break;
                        case "Small":
                            hours += 4;
                            break;
                        case "Medium":
                            hours += 7;
                            break;
                        case "Large":
                            hours += 10;
                            break;
                    }

                    switch ($rootScope.selectedWorkstation.employee.task.contractType) {
                        default:
                            hours += 0;
                            break;
                        case ContractTypes.FixedPrice:
                            hours += 4;
                            break;
                        case ContractTypes.LimitedRFQ:
                            hours += 4;
                            break;
                    }

                    ctrl.hoursToSearch = hours;
                });

                ctrl.startSearch = () => {
                    $rootScope.selectedWorkstation.employee.task.contract = Generators.GenerateContract($rootScope.selectedWorkstation.employee.task.contractType, $rootScope.selectedWorkstation.employee.task.size);
                    let minutes = ctrl.hoursToSearch * 60;

                    $rootScope.selectedWorkstation.employee.task.totalMinutes = minutes;
                    $rootScope.selectedWorkstation.employee.task.completedMinutes = 0;
                    $rootScope.selectedWorkstation.employee.task.state = Enums.TaskStates.Running;

                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                };

                ctrl.showContract = () => {
                    if (!$rootScope.settings.paused) {
                        Game.Lifecycle.PauseTime(true, true);
                        $rootScope.settings.resumeGameWhenClosed = true;
                    }
                    $rootScope.contractRequest = $rootScope.selectedWorkstation.employee.task.contract;
                };

                ctrl.cancel = () => {
                    $rootScope.selectedWorkstation.employee.task.contract = null;
                    $rootScope.selectedWorkstation.employee.task.contractId = null;
                    $rootScope.selectedWorkstation.employee.task.state = Enums.TaskStates.Stopped;
                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                };

                ctrl.clickAutoRepeat = () => {
                    if (ctrl.controlledByManager) {
                        $rootScope.selectedWorkstation.employee.task.autoRepeat = !$rootScope.selectedWorkstation.employee.task.autoRepeat
                    } else {
                        $rootScope.showMessage('', Helpers.GetLocalized('auto_repeat_warning'));
                    }
                };
            },
            controllerAs: 'ctrl'
        }
    });
})();
